#!/usr/bin
# -*- coding: utf-8 -*-

# #############################################################################
#
# Dernières modifications :
# 2025/05/07, Steven MARGUET :
#   Remplacement module load intelmpi/13.2 par module load intelmpi/19.12.320
#   pour résoudre problèmes de parallélisation sur Olympe pour Christophe
#
# 2022/07/08, Steven MARGUET :
#   Mise à jour pour ABAQUS 2022hf1
#
# #############################################################################


# #############################################################################
#
# Programme développé pour le calculateur : O L Y M P E
#
# #############################################################################
# Ce programme génère automatiquement un fichier .cmd
# permettant d'executer un calcul Abaqus.
#
# Il est scindé en quatre parties :
# 1. préparation Abaqus ;
# 2. demande des ressources ;
# 3. génération du fichier SLURM ;
# 4. affichage du fichier SLURM généré
#    et exécution automatique si demandé
#
# Il doit pouvoir s'adapter à d'autres programmes.
# Il est fourni sans aucune garantie d'aucune sorte.
#
# -----------------------------------------------------------------------------
# Utilisation
# 1. Créez un dossier /tmpdir/<votreLogin>/<votreDossier>
# 2. Placer-y le fichier script.py
# 3. Placer-y le fichier <votreFichier>.inp
#    Éventuellement ajoutez-y :
#        - plusieurs autres <fichier>.inp si vous avez des *Include
#        - <votreFichier>.f (je ne sais plus...)
# 4. Placez-vous dans /tmpdir/<votreLogin>/<votreDossier>
#    et tapez la commande : python olympeAbaqus.py
#
# 5. Laissez-vous guider pour l'allocation des ressources
#    (ou pré-configurez le script via les variables en MAJUSCULES
#     un peu plus bas)
#
# => Le programme génere un fichier .cmd que vous pouvez exécuter directement
#    à la fin du programme, ou bien à la main via
#        sbatch fichier_slurm.cmd
#
# => Le fichier_slurm.cmd génère un repertoire
#        /tmpdir/<votreLogin>/<votreDossier>/$PBS_JOBID
#    ou
#        /tmpdir/<votreLogin>/<votreDossier>/
#            USER_RELATIVE_FOLDER_FOR_WORKDIR/$PBS_JOBID
#    (si vous avez defini la variable USER_RELATIVE_FOLDER_FOR_WORKDIR)
#
# => Il y recopie les fichiers nécessaires (.inp, .f, .cmd, ...)
# => Enfin il exécute Abaqus
#
#    Les fichiers de résultats seront dans
#        /tmpdir/<votreLogin>/<votreDossier>/$PBS_JOBID
#    ou
#        /tmpdir/<votreLogin>/<votreDossier>/
#            USER_RELATIVE_FOLDER_FOR_WORKDIR/$PBS_JOBID
#
# 6. Pour vérifier que tout va bien :
#      squeue -u <votreLogin>
#
# /!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\--
#
# ATTENTION : toutes les données d'entrées ne sont pas verifiées
#             (presence effective des fichiers dans les dossiers...).
#             Vous êtes responsable de leur cohérence.
#
# /!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\/!\--
#
# -----------------------------------------------------------------------------
# License GNU/GPL
# Copyright 2012-2020 Calmip, Institut Clément Ader
# #############################################################################



# #############################################################################
# VOCABULAIRE
#
# Un nœud : ensemble de microprocesseurs/sockets partageant leur memoire vive
# Un socket : un microprocesseur
# Un cœur : une unité de calcul (un microprocesseur/socket
#                                peut posseder plusieurs cœurs)
# Une "queue" : un type d'ordre de calcul associer à un intervalle
#               de ressources
# Ram : mémoire vive
# #############################################################################



# Automatisation (À MODIFIER PAR L'UTILISATEUR ================================
#   Remarques :
#   - si une variable est à None, le script demandera une valeur
#     en mode interactif (dans la mesure du possible)
#   - si une variable est à "o" ou "n", le script reagit en conséquence
#
#   Cela permet d'automatiser le lancement sans aucune intervention
#   interactive si desiré.

INP_FILE = None # "nom_fichier.inp"
FORTRAN = "n" # à choisir dans (None, "o", "n")
FORTRAN_FILE = None # "nom_fichier.f"
OTHER_FILES = "n" # à choisir dans (None, "o", "n")
RESTART = "n" # à choisir dans (None, "o", "n")
MACHINE = None # "nom_queue"
MACHINE_CPUS = None # entier
CLEAN = None # à choisir dans (None, "o", "n")
CHANGER_TEMPS = None # à choisir dans (None, "o", "n")
TEMPS_MAX_H = 72 # en heures être humain
AFFICHER = "o" # à choisir dans (None, "o", "n")
EXECUTER = None # à choisir dans (None, "o", "n")


USER_RELATIVE_FOLDER_FOR_WORKDIR = "" # nom du dossier relatif, sans / à la fin !
LOG_PERFORMANCE = "n" # à choisir dans ("o", "n")
LOG_PERFORMANCE_FILE = "log_perfo.txt"



# VERSION D'ABAQUS : à modifier par l'utilisateur =============================
#                    (réglage plus bas niveau...)
E_MAIL = ""
LOGIN = ""

ABAQUS_EXEC = "abq2022hf1" # "abq2018" "abq2016" "abq6135" "abq6112"
ABAQUS_PATH = "/usr/local/abaqus/Commands" # sans / a la fin !
#LICENSE_SERVER = "27000@lic-mixte-abaqus.insa-toulouse.fr" obsolète depuis janvier 2021
LICENSE_SERVER = "27000@ica-abaqus.insa-toulouse.fr"

ENV_FILE = "abaqus_v6.env" # fichier d'environnement modifié
                           # pour indiquer les nœuds de calcul
                           # à Abaqus
MACHINES_FILE = "machinesfile.txt" # fichier utilisé pour récupérer
                                   # la liste des nœuds de calcul
                                   # attribués par SLURM

FICHIERS_A_CONSERVER_SI_NETTOYAGE = (".cmd", ".inp", ".f")
FICHIERS_NECESSAIRES_POUR_RESTART = (".odb", ".res", ".abq", ".mdl",
                                     ".pac", ".stt", ".prt", ".sel")
LARGEUR_TERMINAL = 80
ABAQUS_JOB = "explicit" # à choisir dans ("explicit", "standard")
ABAQUS_DIR = {"abq2022hf1": "abaqus2022hf1",
              "abq2018": "abaqus2018",
              "abq2016": "abaqus2016",
              "abq6135": "6.13-5",
              "abq6112": "6.11-2"}
ABAQUS_ENV = {"abq2022hf1": "/usr/local/abaqus/SIMULIA/EstProducts/2022/linux_a64/SMA/site/",
              "abq2018": "/usr/local/abaqus/abaqus2018/SimulationServices/V6R2018x/linux_a64/SMA/site",
              "abq2016": "/usr/local/abaqus/abaqus2016/linux_a64/SMA/site",
              "abq6135": "/usr/local/abaqus/6.13-5/SMA/site",
              "abq6112": "/usr/local/abaqus/6.11-2/site"}
ABAQUS_MOD = {"abq2022hf1": ("module purge", "module load abaqus/2022", "module load intel/09.1","module load intelmpi/19.12.320", "module load intel/12.1.5"),
              "abq2018": ("module purge", "module load intel/09.1","module load intelmpi/13.2", "module load intel/12.1.5"),
              "abq2016": ("module purge", "module load intel/09.1","module load intelmpi/13.2", "module load intel/12.1.5"),
              "abq6135": ("module purge", "module load intel/09.1","module load intelmpi/13.2", "module load intel/12.1.5"),
              "abq6112": ("module purge", "module load intel/09.1","module load intelmpi/13.2", "module load intel/12.1.5")}
#ABAQUS_DIR = ABAQUS_DIR[ABAQUS_EXEC] inutilisées, définies au cas où...
#ABAQUS_ENV = ABAQUS_ENV[ABAQUS_EXEC]



# Constantes ==================================================================
# tuple pour conserver l'ordre (modifié ultérieurement en dictionnaire)
coresForMesca = range(0, 129, 16)
coresForMesca[0] += 1

QUEUES_LIST = (("mono", {"machine":"olympe",
                         "noeuds":(1,),
                         "coeursParSocket":18,
                         "socketsParNoeud":2,
                         "coeursSelectionnables":range(1, 19, 1),
                         "ramParSocket":90.0, # en GB - 32 en vrai mais on prend de la marge
                         "exclusif":"non",
                         "hyperthreading":"oui",
                         "maxTempsCalculHumain":400,
                         "jobParUtilisateur":5}),
               ("noeud", {"machine":"olympe",
                          "noeuds":(1,),
                          "coeursParSocket":18,
                          "socketsParNoeud":2,
                          "coeursSelectionnables":(36,),
                          "ramParSocket":90.0, # en GB
                          "exclusif":"oui",
                          "hyperthreading":"oui",
                          "maxTempsCalculHumain":250,
                          "jobParUtilisateur":5}),
               ("noeud5", {"machine":"olympe",
                           "noeuds":range(2, 6, 1),
                           "coeursParSocket":18,
                           "socketsParNoeud":2,
                           "coeursSelectionnables":range(72, 181, 36),
                           "ramParSocket":90.0, # en GB
                           "exclusif":"oui",
                           "hyperthreading":"oui",
                           "maxTempsCalculHumain":150,
                           "jobParUtilisateur":2}),
               ("noeud10", {"machine":"olympe",
                            "noeuds":range(6, 11, 1),
                            "coeursParSocket":18,
                            "socketsParNoeud":2,
                            "coeursSelectionnables":range(216, 361, 36),
                            "ramParSocket":90.0, # en GB
                            "exclusif":"oui",
                            "hyperthreading":"oui",
                            "maxTempsCalculHumain":110,
                            "jobParUtilisateur":2}),
               ("noeud20", {"machine":"olympe",
                            "noeuds":range(11, 21, 1),
                            "coeursParSocket":18,
                            "socketsParNoeud":2,
                            "coeursSelectionnables":range(396, 721, 36),
                            "ramParSocket":90.0, # en GB
                            "exclusif":"oui",
                            "hyperthreading":"oui",
                            "maxTempsCalculHumain":75,
                            "jobParUtilisateur":1}),
               ("noeud40", {"machine":"olympe",
                            "noeuds":range(21, 41, 1),
                            "coeursParSocket":18,
                            "socketsParNoeud":2,
                            "coeursSelectionnables":range(756, 1441, 36),
                            "ramParSocket":90.0, # en GB
                            "exclusif":"oui",
                            "hyperthreading":"oui",
                            "maxTempsCalculHumain":36,
                            "jobParUtilisateur":1}),
               ("noeud50", {"machine":"olympe",
                            "noeuds":range(41, 51, 1),
                            "coeursParSocket":18,
                            "socketsParNoeud":2,
                            "coeursSelectionnables":range(1476, 1801, 20),
                            "ramParSocket":90.0, # en GB
                            "exclusif":"oui",
                            "hyperthreading":"oui",
                            "maxTempsCalculHumain":24,
                            "jobParUtilisateur":1}),
               ("visu", {"machine":"visu",
                         "noeuds":range(1,),
                         "coeursParSocket":18,
                         "socketsParNoeud":2,
                         "coeursSelectionnables":range(1, 37, 1),
                         "ramParSocket":180.0, # en GB
                         "exclusif":"non",
                         "hyperthreading":"oui",
                         "maxTempsCalculHumain":4,
                         "jobParUtilisateur":1}),
               ("mesca", {"machine":"mesca",
                          "noeuds":range(1,),
                          "coeursParSocket":16,
                          "socketsParNoeud":8,
                          "coeursSelectionnables":coresForMesca,
                          "ramParSocket":256.0, # en GB
                          "exclusif":"oui",
                          "hyperthreading":"non",
                          "maxTempsCalculHumain":4,
                          "jobParUtilisateur":1})
                         )

GIGA_2_MEGA = 1024
DELTA_RAM_PAR_COEUR = 0.1 # nombre de GB de RAM à enlever à la valeur réelle
                          # par cœur pour éviter les problèmes de conversion


# Fonction utilitaire pour générer le fichier scriptEnv.py automatiquement ----
def genScriptEnv(machinesFile, envFile, nCoeursParNoeud):

    # msgSE représente le contenu du fichier scriptEnv.py
    # sous forme d'une chaîne de caractères.
    # sciptEnv.py :
    # - récupère le contenu du fichier machinesFile
    # - génère le fichier abaqus_v6.env modifié pour nos besoins

    msgSE = "#!/usr/bin\n"
    msgSE += "# -*- coding: utf-8 -*-\n"
    msgSE += "\n"

    msgSE += "import os\n"
    msgSE += "cwd = os.getcwd()\n"
    msgSE += "\n"

    # Récupération des nœuds de calcul et création
    # de la variable hostline qui est une chaîne de caractères
    # contenant une ligne du fichier abaqus_v6.env
    # hostline = « mp_host_list=[["noeud1"],["noeud2"]] »...
    msgSE += "hosts = []\n"
    msgSE += "with open(\"" + machinesFile + "\", \"r\") as hostFile:\n"
    msgSE += "    while True:\n"
    msgSE += "        line = hostFile.readline()\n"
    msgSE += "        if not line:\n"
    msgSE += "            break\n"
    msgSE += "        hosts.append(line[:-1])\n"
    msgSE += "\n"

    msgSE += "if len(hosts) > 1: # ne fonctionne pas sinon\n"
    msgSE += "    hostline = \"mp_host_list=[\"\n"
    msgSE += "    for host in hosts:\n"
    msgSE += "        hostline += \"[\'\" + host + \"\', "
    msgSE += str(nCoeursParNoeud) + "],\"\n"
    msgSE += "    hostline = hostline[:-1] + \"]\\n\"\n\n"

    # Création de la chaîne de caractères qui contient le contenu
    # modifié du fichier abaqus_v6.env
    # Cette chaîne de caractères est elle même encapsulée dans
    # la chaîne de caractères liée à scriptEnv.py
    # Attention, il faut être bien accroché pour la ponctuation :
    # - pour écrire un " dans une sous chaîne : \"
    # -                '                        \'
    # -                \n                       \\n
    msgSE += "msg = \"# abaqus V6 Environment File - ICA version\\n\"\n"
    msgSE += "msg += \"# This file is used to import other environment files located in the site\\n\"\n"
    msgSE += "msg += \"# directory. Custom parameters can be added here, but it is recommended to\\n\"\n"
    msgSE += "msg += \"# use the custom_v6.env file instead. - Ca, ca ne fonctionne pas.\\n\\n\"\n"

    msgSE += "msg += \"import os, re, glob, driverUtils\\n\\n\"\n"
    msgSE += "msg += \"# Utility function for loading site environment files\\n\"\n"
    msgSE += "msg += \"def importEnv(name, ldict=locals(), gdict=globals()):\\n\"\n"
    msgSE += "msg += \"    from driverUtils import locateSite, locateFile\\n\"\n"
    msgSE += "msg += \"    env = locateFile(locateSite(), \'\', name)\\n\"\n"
    msgSE += "msg += \"    if env:\\n\"\n"
    msgSE += "msg += \"        execfile(env, gdict, ldict)\\n\"\n"
    msgSE += "msg += \"    else:\\n\"\n"
    msgSE += "msg += \"        print \'Cannot locate environment file: \', name\\n\\n\"\n"

    # Contenu du basic_v6.env en dur (ajouté au abaqus_v6.env à l'endroit où l'import se fait usuellement)
    msgSE += "msg += \"standard_parallel = ALL\\n\"\n"
    msgSE += "msg += \"mp_mode = MPI\\n\"\n"
    msgSE += "msg += \"mp_file_system = (DETECT,DETECT)\\n\"\n"
    msgSE += "msg += \"mp_num_parallel_ftps = (4, 4)\\n\"\n"
    msgSE += "msg += \"mp_environment_export = (\'MPI_PROPAGATE_TSTP\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_CM_BUFFERING\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_CM_BUFFERING_LIMIT\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_ITERATIVE_SOLVER_VERBOSE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_DMPSOLVER_BWDPARALLELOFF\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_ELP_SURFACE_SPLIT\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_ELP_SUSPEND\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_HOME\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_MEMORY_MODE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_MPI_MESSAGE_TRACKING\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_MPI_VERBOSE_LEVEL\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_STD_ALLOW_SURFACE_TO_BEAM\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_PATH\',\\n\"\n"
    msgSE += "msg += \"                         \'ABAQUS_CSE_RELTIMETOLERANCE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_RESOURCE_MONITOR\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_RESOURCE_USEMALLINFO\',\\n\"\n"
    msgSE += "msg += \"                         \'ABAQUS_LANG\',\\n\"\n"
    msgSE += "msg += \"                         \'ABAQUS_CSE_CURRCONFIGMAPPING\',\\n\"\n"
    msgSE += "msg += \"                         \'ABAQUS_MPF_DIAGNOSTIC_LEVEL\',\\n\"\n"
    msgSE += "msg += \"                         \'ABAQUSLM_LICENSE_FILE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_CRTMALLOC\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_DATACHECK\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_RECOVER\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_RESTART\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_SPLITFILE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_XPL_WINDOWDUMP\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQ_XPL_PARTITIONSIZE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQLMHANGLIMIT\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQLMQUEUE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABQLMUSER\',\\n\"\n"
    msgSE += "msg += \"                         \'CCI_RENDEZVOUS\',\\n\"\n"
    msgSE += "msg += \"                         \'DOMAIN\',\\n\"\n"
    msgSE += "msg += \"                         \'DOMAIN_CPUS\',\\n\"\n"
    msgSE += "msg += \"                         \'DOUBLE_PRECISION\',\\n\"\n"
    msgSE += "msg += \"                         \'FLEXLM_DIAGNOSTICS\',\\n\"\n"
    msgSE += "msg += \"                         \'FOR0006\',\\n\"\n"
    msgSE += "msg += \"                         \'FOR0064\',\\n\"\n"
    msgSE += "msg += \"                         \'FOR_IGNORE_EXCEPTIONS\',\\n\"\n"
    msgSE += "msg += \"                         \'FOR_DISABLE_DIAGNOSTIC_DISPLAY\',\\n\"\n"
    msgSE += "msg += \"                         \'LD_PRELOAD\',\\n\"\n"
    msgSE += "msg += \"                         \'MP_NUMBER_OF_THREADS\',\\n\"\n"
    msgSE += "msg += \"                         \'MPC_GANG\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_FLAGS\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_FLUSH_FCACHE\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_RDMA_NENVELOPE\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_SOCKBUFSIZE\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_USE_MALLOPT_MMAP_MAX\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_USE_MALLOPT_MMAP_THRESHOLD\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_USE_MALLOPT_SBRK_PROTECTION\',\\n\"\n"
    msgSE += "msg += \"                         \'MPI_WORKDIR\',\\n\"\n"
    msgSE += "msg += \"                         \'MPCCI_DEBUG\',\\n\"\n"
    msgSE += "msg += \"                         \'MPCCI_CODEID\',\\n\"\n"
    msgSE += "msg += \"                         \'MPCCI_JOBID\',\\n\"\n"
    msgSE += "msg += \"                         \'MPCCI_NETDEVICE\',\\n\"\n"
    msgSE += "msg += \"                         \'MPCCI_TINFO\',\\n\"\n"
    msgSE += "msg += \"                         \'MPCCI_SERVER\',\\n\"\n"
    msgSE += "msg += \"                         \'MPIEXEC_AFFINITY_TABLE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABAQUS_CCI_DEBUG\',\\n\"\n"
    msgSE += "msg += \"                         \'NCPUS\',\\n\"\n"
    msgSE += "msg += \"                         \'OMP_DYNAMIC\',\\n\"\n"
    msgSE += "msg += \"                         \'OMP_NUM_THREADS\',\\n\"\n"
    msgSE += "msg += \"                         \'OUTDIR\',\\n\"\n"
    msgSE += "msg += \"                         \'PAIDUP\',\\n\"\n"
    msgSE += "msg += \"                         \'PARALLEL_METHOD\',\\n\"\n"
    msgSE += "msg += \"                         \'RAIDEV_NDREG_LAZYMEM\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_SYMBOLIC_GENERALCOLLAPSE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_SYMBOLIC_GENERAL_MAXCLIQUERANK\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_ADM_MINIMUMINCREASE\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_ADM_MINIMUMDECREASE\',\\n\"\n"
    msgSE += "msg += \"                         \'IPATH_NO_CPUAFFINITY\',\\n\"\n"
    msgSE += "msg += \"                         \'MALLOC_MMAP_THRESHOLD_\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_EXT_SIMOUTPUT\',\\n\"\n"
    msgSE += "msg += \"                         \'SMA_WS\',\\n\"\n"
    msgSE += "msg += \"                         \'SMA_PARENT\',\\n\"\n"
    msgSE += "msg += \"                         \'SMA_PLATFORM\',\\n\"\n"
    msgSE += "msg += \"                         \'ABA_PRE_DECOMPOSITION\',\\n\"\n"
    msgSE += "msg += \"                         \'ACML_FAST_MALLOC\',\\n\"\n"
    msgSE += "msg += \"                         \'ACML_FAST_MALLOC_CHUNK_SIZE\',\\n\"\n"
    msgSE += "msg += \"                         \'ACML_FAST_MALLOC_MAX_CHUNKS\',\\n\"\n"
    msgSE += "msg += \"                         \'ACML_FAST_MALLOC_DEBUG\',\\n\"\n"
    msgSE += "msg += \"                         \'MKL_NUM_THREADS\',\\n\"\n"
    msgSE += "msg += \"                         \'MKL_DYNAMIC\')\\n\\n\"\n"

    # Ajout pour Joël
    msgSE += "msg += \"os.environ[\'ABQ_XPL_MSPEI\'] = \'ON\'\\n\"\n"
    msgSE += "msg += \"os.environ[\'ABA_XPL_DOMAINTIMERS\'] = \'ON\'\\n\\n\"\n"

    # Contenu du lnx86_64_v6.env en dur (ajouté au abaqus_v6.env à l'endroit où l'import se fait usuellement)
    msgSE += "msg += \"# Import platform specific parameters such as compiler and MPI settings\\n\"\n"
    msgSE += "msg += \"platform = driverUtils.getPlatform()\\n\\n\"\n"

    msgSE += "msg += \"#MPI implementation handling\\n\"\n"
    msgSE += "msg += \"mpiCppImpl = \'\'\\n\"\n"
    msgSE += "msg += \"mp_rsh_command = \'ssh -n -l %U %H %C\'\\n\"\n"
    msgSE += "msg += \"#mp_mpi_implementation = PMPI #Platform MPI\\n\"\n"
    msgSE += "msg += \"mp_mpi_implementation = IMPI #<--- Uncomment this line and comment out prior definition to switch to IMPI (Intel)\\n\"\n"
    msgSE += "msg += \"#mp_mpi_implementation = CMPI #<--- Uncomment this line and comment out prior definition to switch to CMPI (Cray)\\n\"\n"
    msgSE += "msg += \"if mp_mpi_implementation == PMPI:\\n\"\n"
    msgSE += "msg += \"    pmpipath = driverUtils.locateFile(os.environ.get(\'ABA_PATH\', \'\'), \'pmpi-9.1.2/bin\', \'mpirun\')\\n\"\n"
    msgSE += "msg += \"    mp_mpirun_path = {PMPI: pmpipath}\\n\"\n"
    msgSE += "msg += \"    mpiCppImpl = \'-DABQ_MPI_PMPI\'\\n\"\n"
    msgSE += "msg += \"    mpiLnkImpl = \'-lmpi\'\\n\"\n"
    msgSE += "msg += \"if mp_mpi_implementation == IMPI:\\n\"\n"
    msgSE += "msg += \"    impipath = driverUtils.locateFile(os.environ.get(\'ABA_PATH\', \'\'), \'impi/bin\', \'mpiexec.hydra\')\\n\"\n"
    msgSE += "msg += \"    mp_mpirun_path = {IMPI: impipath}\\n\"\n"
    msgSE += "msg += \"    mpiCppImpl = \'-DABQ_MPI_IMPI\'\\n\"\n"
    msgSE += "msg += \"    mpiLnkImpl = \'-L/usr/local/intel_new/impi/5.0.1.035/intel64/lib -lmpi -lmpifort\'\\n\"\n"
    msgSE += "msg += \"if mp_mpi_implementation == CMPI:\\n\"\n"
    msgSE += "msg += \"    cmpipath = driverUtils.locateFile(os.environ.get(\'ABA_PATH\', \'\'), \'impi/bin\', \'mpiexec.hydra\')\\n\"\n"
    msgSE += "msg += \"    mp_mpirun_path = {CMPI: cmpipath}\\n\"\n"
    msgSE += "msg += \"    mpiCppImpl = \'-DABQ_MPI_CMPI\'\\n\"\n"
    msgSE += "msg += \"    mpiLnkImpl = \'-limpi -lmpifort\'\\n\"\n"
    msgSE += "msg += \"    mp_file_system = (SHARED, SHARED)\\n\\n\"\n"

    msgSE += "msg += \"fortCmd = \'ifort\'   # <-- Fortran compiler\\n\"\n"
    msgSE += "msg += \"cppCmd  = \'g++\'     # <-- C++ compiler\\n\\n\"\n"

    msgSE += "msg += \"# Avoid signal trapping by the Fortran RTE\\n\"\n"
    msgSE += "msg += \"os.environ[\'FOR_IGNORE_EXCEPTIONS\'] = \'1\'\\n\"\n"
    msgSE += "msg += \"# Disable messages from the Fotran RTE\\n\"\n"
    msgSE += "msg += \"os.environ[\'FOR_DISABLE_STACK_TRACE\'] = \'1\'\\n\\n\"\n"

    msgSE += "msg += \"# Do not let QLogic InfiniPath driver set processor affinity.\\n\"\n"
    msgSE += "msg += \"os.environ[\'IPATH_NO_CPUAFFINITY\'] = \'1\'\\n\\n\"\n"

    msgSE += "msg += \"# Add the flag \'-free\' to the compile_fortran command below to use free-format FORTRAN 90 syntax.\\n\\n\"\n"

    msgSE += "msg += \"compile_fortran = [fortCmd,\\n\"\n"
    msgSE += "msg += \"                   \'-V\',\\n\"\n"
    msgSE += "msg += \"                   \'-c\', \'-fPIC\', \'-auto\', \'-mP2OPT_hpo_vec_divbyzero=F\', \'-extend_source\',\\n\"\n"
    msgSE += "msg += \"                   \'-fpp\', \'-WB\', \'-I%I\']\\n\\n\"\n"

    msgSE += "msg += \"# Additional command-line options for the Intel C/C++ Compilers:\\n\"\n"
    msgSE += "msg += \"# \'-cxxlib\', \'-Kc++eh\', \'-Krtti\', \'-Kc++\', \'-pc64\', \'-restrict\', \'-i-dynamic\',\\n\"\n"
    msgSE += "msg += \"# \'-we1011\', \'-we120\',  \'-we117\', \'-we556\', \'-we144\', \'-we268\', \'-we1224\', \'-we167\', \'-we880\'\\n\\n\"\n"

    msgSE += "msg += \"compile_cpp = [cppCmd,\\n\"\n"
    msgSE += "msg += \"               \'-c\', \'-fPIC\', \'-w\', \'-Wno-deprecated\', \'-DTYPENAME=typename\',\\n\"\n"
    msgSE += "msg += \"               \'-D_LINUX_SOURCE\', \'-DABQ_LINUX\', \'-DABQ_LNX86_64\', \'-DSMA_GNUC\',\\n\"\n"
    msgSE += "msg += \"               \'-DFOR_TRAIL\', \'-DHAS_BOOL\', \'-DASSERT_ENABLED\',\\n\"\n"
    msgSE += "msg += \"               \'-D_BSD_TYPES\', \'-D_BSD_SOURCE\', \'-D_GNU_SOURCE\',\\n\"\n"
    msgSE += "msg += \"               \'-D_POSIX_SOURCE\', \'-D_XOPEN_SOURCE_EXTENDED\', \'-D_XOPEN_SOURCE\',\\n\"\n"
    msgSE += "msg += \"               \'-DHAVE_OPENGL\', \'-DHKS_OPEN_GL\',  \'-DGL_GLEXT_PROTOTYPES\',\\n\"\n"
    msgSE += "msg += \"               \'-DMULTI_THREADING_ENABLED\', \'-D_REENTRANT\',\\n\"\n"
    msgSE += "msg += \"               \'-DABQ_MPI_SUPPORT\', \'-DBIT64\', \'-D_LARGEFILE64_SOURCE\', \'-D_FILE_OFFSET_BITS=64\', \'%P\',\\n\"\n"
    msgSE += "msg += \"               mpiCppImpl,\\n\"\n"
    msgSE += "msg += \"               # \'-O0\', # <-- Optimization level\\n\"\n"
    msgSE += "msg += \"               # \'-g\',  # <-- Debug symbols\\n\"\n"
    msgSE += "msg += \"               \'-I%I\']\\n\\n\"\n"

    msgSE += "msg += \"compile_fmu = [cppCmd,\\n\"\n"
    msgSE += "msg += \"               \'-c\', \'-fPIC\',\'-I%I\']\\n\\n\"\n"

    msgSE += "msg += \"link_fmu = [cppCmd,\\n\"\n"
    msgSE += "msg += \"            \'-fPIC\', \'-shared\', \'-o\', \'%J\', \'%M\']\\n\\n\"\n"

    msgSE += "msg += \"link_sl = [fortCmd,\\n\"\n"
    msgSE += "msg += \"           \'-V\',\\n\"\n"
    msgSE += "msg += \"           \'-cxxlib\', \'-fPIC\', \'-threads\', \'-shared\',\\n\"\n"
    msgSE += "msg += \"           \'%E\', \'-Wl,-soname,%U\', \'-o\', \'%U\', \'%F\', \'%A\', \'%L\', \'%B\', \'-parallel\',\\n\"\n"
    msgSE += "msg += \"           \'-Wl,-Bdynamic\', \'-i-dynamic\', \'-lifport\', \'-lifcoremt\', \'-lsvml\', \'-lirc\', mpiLnkImpl]\\n\\n\"\n"

    msgSE += "msg += \"link_exe = [cppCmd,\\n\"\n"
    msgSE += "msg += \"            \'-fPIC\',\\n\"\n"
    msgSE += "msg += \"            \'-Wl,-Bdynamic\', \'-o\', \'%J\', \'%F\', \'%M\', \'%L\', \'%B\', \'%O\', \'-lpthread\', \'-limf\', \'-lifcoremt\', \'-lsvml\', \'-lirc\']\\n\\n\"\n"

    msgSE += "msg += \"# Remove the temporary names from the namespace\\n\"\n"
    msgSE += "msg += \"del cppCmd\\n\"\n"
    msgSE += "msg += \"del fortCmd\\n\"\n"
    msgSE += "msg += \"del mpiCppImpl\\n\"\n"
    msgSE += "msg += \"del mpiLnkImpl\\n\\n\"\n"
    msgSE += "msg += \"if mp_mpi_implementation == PMPI:\\n\"\n"
    msgSE += "msg += \"    del pmpipath\\n\"\n"
    msgSE += "msg += \"if mp_mpi_implementation == IMPI:\\n\"\n"
    msgSE += "msg += \"    del impipath\\n\"\n"
    msgSE += "msg += \"if mp_mpi_implementation == CMPI:\\n\"\n"
    msgSE += "msg += \"    del cmpipath\\n\\n\"\n"

    # Contenu du custom_v6.env en dur
    msgSE += "msg += \"license_server_type=FLEXNET\\n\"\n"
    msgSE += "msg += \"abaquslm_license_file="
    msgSE += "\'" + LICENSE_SERVER + "\'\\n"
    msgSE += "\"\n"
    msgSE += "msg += \"academic=RESEARCH\\n\"\n\n"

    msgSE += "if len(hosts) > 1:\n"
    msgSE += "    msg += hostline + \"\\n\"\n\n"

    msgSE += "msg += \"del importEnv, driverUtils, platform\\n\\n\"\n\n"

    msgSE += "with open(\"" + envFile + "\", \"w\") as outputFile:\n"
    msgSE += "    outputFile.write(msg)\n"

    with open("scriptEnv.py", "w") as myFile:
        myFile.write(msgSE)


# =============================================================================
# PREMIERE PARTIE : ABAQUS
# =============================================================================

# Verification préliminaire de la présence d'un fichier d'entrée --------------
import os
import sys

inputFiles = []
fortranFiles = []

inputFile = False
abaqusEnvFile = False

cwd = os.getcwd()
files = os.listdir(cwd)
for file_ in files:
    if file_[-4:].lower() == ".inp":
        inputFile = True
        inputFiles.append(file_)
    elif file_[-2:].lower() == ".f":
        fortranFiles.append(file_)


# Pour lecture directe du fichier INP à partir du terminal --------------------
print inputFiles
if len(sys.argv) > 1:
    INP_FILE = sys.argv[1]
    if not (INP_FILE in inputFiles):
        print "Le fichier .inp spécifié en ligne de commande (" + \
              inp + ") n'est pas dans le dossier."

    if len(sys.argv) > 2:
        for extraFile in sys.argv[2:]:
            if not (extraFile in inputFiles):
                print "Le fichier " + extraFile +  " spécifié en ligne de commande " + \
                      "n'est pas dans le dossier."

if not inputFile:
    print "Il doit y avoir un fichier d'entrée .inp dans le dossier %s" \
           % (cwd,)
    exit(1)


# Fichier .inp
print "\n"
print "=" * LARGEUR_TERMINAL
print "Sélection des fichiers d'entree, nettoyage eventuel"
print "=" * LARGEUR_TERMINAL

if INP_FILE in inputFiles:
    inputFileName = INP_FILE
    print "Fichier d'entree : " + inputFileName
else:
    msg = "\nFichier d'entree (.inp) :\n  [ "
    for ifile in inputFiles:
        msg += ifile + " ; "
    msg = msg[:-2] + "]\n>>> "

    inputFileExists = False
    while not inputFileExists:
        inputFileName = raw_input(msg)
        inputFileExists = os.path.isfile(inputFileName)
        if not inputFileExists:
            print "Le fichier specifie n'existe pas..."
        if not inputFileName[-4:] == ".inp":
            print "Le fichier doit avoir '.inp' pour extension..."

a = len(inputFileName) - 4


# Fichier pour programme utilisateur (Fortran)
if FORTRAN == "o":
    choix = "o"
    fortranFile = FORTRAN_FILE
elif FORTRAN == "n":
    choix = "n"
    fortranFile = None
else:
    fortranFile = False
    msg =  "\nSouaitez-vous spécifier un fichier Fortran (o/n) ?\n>>> "
    choix = "a"
    while not choix.lower() in ("o", "n"):
        choix = raw_input(msg)
        choix = choix.lower()

if choix == "o":
    if FORTRAN == "o":
        fortranFileName = FORTRAN_FILE
        fortranFileExists = True
    else:
        fortranFile = True
        fortranFileExists = False

    if len(fortranFiles) == 0:
        print "\nIl n'y a pas de fichier .f dans le dossier :" + \
               "\n  %s" % (cwd,)
        exit(1)

    msg = "\nFichier Fortran (.f) :\n  [ "
    for ffile in fortranFiles:
        msg += ffile + " ; "
    msg = msg[:-2] + "]\n>>> "

    while not fortranFileExists:
        fortranFileName = raw_input(msg)
        fortranFileExists = os.path.isfile(fortranFileName)
        if not fortranFileExists:
            print "Le fichier specifie n'existe pas..."
        if not fortranFileName[-2:] == ".f":
            print "Le fichier doit avoir '.f' pour extension..."


# Fichiers complémentaires
otherFilesToCopy = []
if OTHER_FILES == "o":
    choix = "o"
    otherFiles = OTHER_FILES
elif OTHER_FILES == "n":
    choix = "n"
    otherFiles = None
else:
    otherFiles = False
    msg =  "\nSouaitez-vous copier d'autres fichiers (o/n) ?\n>>> "
    choix = "a"
    while not choix.lower() in ("o", "n"):
        choix = raw_input(msg)
        choix = choix.lower()

if choix == "o":
    anotherOne = True
    while anotherOne:
        otherFileName = raw_input("Saisissez le nom du fichier à copier : ")

        if not os.path.isfile(otherFileName):
            print("Le fichier n'existe pas, veuillez en saisir un correct.")
            pass
        else:
            otherFilesToCopy.append(otherFileName)
            choice = "?"
            print "\nSouhaitez-vous copier d'autres fichiers (o/n) ?"
            while not choice.lower() in ("o", "n"):
                choice = raw_input(">>> ")
            if choice == "n":
                anotherOne = False


# Restart
if RESTART == "o":
    choix = "o"
elif RESTART == "n":
    choix = "n"
else:
    msg = "\nCe calcul est-il un 'RESTART' d'un precedent calcul (o/n) ?\n>>> "
    choix = None
    while not choix in ("o", "n"):
        choix = raw_input(msg)
        choix = choix.lower()

if choix == "o":
    restart = True
    print "\n" + "*" * LARGEUR_TERMINAL
    print "ATTENTION : vous devez impérativement choisir le même nombre" + \
           " de coeurs que pour le calcul initial."
    print "*" * LARGEUR_TERMINAL
    FAC = FICHIERS_A_CONSERVER_SI_NETTOYAGE \
        + FICHIERS_NECESSAIRES_POUR_RESTART
else:
    restart = False
    FAC = FICHIERS_A_CONSERVER_SI_NETTOYAGE



# Nettoyage
files = os.getcwd()
files = os.listdir(files)
filesToRemove = []
for file in files:
    if file[:a] == inputFileName[:-4] and \
       file[-4:] not in FAC:
        filesToRemove.append(file)

if CLEAN == "o":
    if len(filesToRemove) > 0:
        if len(filesToRemove) == 1:
            msg = "\nVoulez-vous supprimer le fichier :"
        else:
            msg = "\nVoulez-vous supprimer les fichiers (o/n) ?"
        for file in filesToRemove:
            msg += "\n- " + file
        msg += "\n>>> "

        clean = None
        while not clean in ("o","n"):
            clean = raw_input(msg)
            clean = clean.lower()

        if clean == "o":
            for file in filesToRemove:
                os.remove(file)
                print "- " + file + " supprimé"




# =============================================================================
# SECONDE PARTIE : DEMANDE DES RESSOURCES
# =============================================================================
print "\n"
print "=" * LARGEUR_TERMINAL
print "Reservation des ressources"
print "=" * LARGEUR_TERMINAL
print "\n"



print "+-----------+------+------+------+------+------------+-----------+-----+-----+"
print "|>>>      VERSION MODIFIÉE POUR UTILISER TOUS LES COEURS D'UN NOEUD       <<<|"
print "+-----------+------+------+------+------+------------+-----------+-----+-----+"
print "|     queue | minN | maxN | minC | maxC | ram/C (Gb) | maxWT (h) |   E |  HT |"
print "+-----------+------+------+------+------+------------+-----------+-----+-----+"
for key, val in QUEUES_LIST:
    minNoeuds = min(val["noeuds"])
    maxNoeuds = max(val["noeuds"])
    minCoeurs = min(val["coeursSelectionnables"])
    maxCoeurs = max(val["coeursSelectionnables"])
    ramParCoeur = val["ramParSocket"] // val["coeursParSocket"]

    print "| %9s | %4i | %4i | %4i | %4i | %10i | %9i | %3s | %3s |" % \
          (key,
           minNoeuds, maxNoeuds,
           minCoeurs, maxCoeurs,
           ramParCoeur, val["maxTempsCalculHumain"],
           val["exclusif"], val["hyperthreading"])
print "+-----------+------+------+------+------+------------+-----------+-----+-----+"

del(minNoeuds, maxNoeuds, minCoeurs, maxCoeurs)

# Transformation du tuple en dictionnaire
QUEUES = {}
for key, val in QUEUES_LIST:
    QUEUES[key] = val


# Type de queue
if MACHINE:
    queue = MACHINE
else:
    msg = "\nQuel type de calcul souhaitez-vous lancer ("
    for key in QUEUES_LIST:
        msg += key[0] + "/"
    msg = msg[:-1] + ") ?\n>>> "
    queue = None
    while not queue in QUEUES.keys():
        queue = raw_input(msg)


jobtype = queue
queue = QUEUES[queue]

minNoeuds = min(queue["noeuds"])
maxNoeuds = max(queue["noeuds"])
minCoeurs = min(queue["coeursSelectionnables"])
maxCoeurs = max(queue["coeursSelectionnables"])
ramParSocket = queue["ramParSocket"]
coeursParSocket = queue["coeursParSocket"]
socketsParNoeud = queue["socketsParNoeud"]
coeursSelectionnables = queue["coeursSelectionnables"]
tempsMax = queue["maxTempsCalculHumain"]

coeursParNoeud = coeursParSocket * socketsParNoeud
ramParNoeud = ramParSocket * socketsParNoeud
ramParCoeur = float(ramParSocket) / coeursParSocket


# Nombre de cœurs -------------------------------------------------------------
if MACHINE_CPUS:
    coeurs = MACHINE_CPUS
else:
    coeurs = None
    if jobtype == "mono":
        msg = "\nDe combien de coeurs physiques souhaitez-vous disposer ?"
        msg += "\nVous pouvez choisir entre " + str(minCoeurs)
        msg += " et " + str(maxCoeurs) + " cœurs :"
        msg += "\n>>> "

        while not coeurs in range(minCoeurs, maxCoeurs+1, 1):
            coeurs = raw_input(msg)
            try:
                coeurs = int(coeurs)
            except:
                pass

    elif jobtype == "noeud":
        nnn = coeursSelectionnables[-1]
        msg = "\nAvec le type de calcul 'noeud',"
        msg += " vous disposez de " + str(nnn) + " coeurs physiques."
        print msg
        coeurs = nnn

    else:
        msg = "\nDe combien de coeurs physiques souhaitez-vous disposer ?"
        msg += "\nVous pouvez choisir parmis :\n  "

        for i, n in enumerate(coeursSelectionnables):
            msg += str(n) + "/"
            if (i+1) % 10 == 0:
                msg += "\n  "
        msg = msg[:-1] + "\nce qui revient a utiliser "
        if jobtype in ("uvprod", "uvcalmip"):
            msg += "tous les coeurs d'un socket"
        else:
            msg += "tous les coeurs de tous les sockets d'un noeud"
        msg += "\n>>> "

        while not coeurs in coeursSelectionnables:
            coeurs = raw_input(msg)
            try:
                coeurs = int(coeurs)
            except:
                pass


# Nombre de nœuds et de sockets -----------------------------------------------
sockets = coeurs // coeursParSocket

if coeurs % coeursParSocket != 0:
    sockets += 1

noeuds = coeurs // coeursParNoeud
if coeurs % coeursParNoeud != 0:
    noeuds += 1

msg = "\nVous allez utiliser :\n" \
    + "- %i noeuds ;\n" % (noeuds,) \
    + "- %i sockets ;\n" % (sockets,) \
    + "- %i coeurs.\n" % (coeurs,)
print msg


# Mémoire vive ----------------------------------------------------------------
ramGlobale = coeurs * ramParCoeur # calcul réalisé de cette manière pour être
                                  # adapté à la queue mono

msg = "\nVous disposez actuellement :\n" \
    + "- de %i Gb de memoire globale ;\n" % (ramGlobale,) \
    + "- soit %i Gb de memoire par noeud ;\n" % (ramParNoeud,) \
    + "- soit %i Gb de memoire par socket ;\n" % (ramParSocket,) \
    + "- soit %i Gb de memoire par coeur." % (ramParCoeur,)



# Temps humain ----------------------------------------------------------------
if CHANGER_TEMPS == "o":
    if not isinstance(TEMPS_MAX_H, int):
        msg = "\nVous disposez actuellement de %i heures 'temps humain' de calcul." \
            % (tempsMax,)
        msg += "\n\nSouhaitez-vous reduire ce temps (o/n) ?\n>>> "
        choix = None
        while not choix in ("o","n"):
            choix = raw_input(msg)
            choix = choix.lower()

        if choix == "o":
            msg = "\nDe combien de temps souhaitez-vous disposer"
            msg += " (le temps doit être inferieur ou egal a"
            msg += " %i heures)" % (tempsMax,)
            msg += " ?\n>>> "

            temps = tempsMax + 1
            while not temps <= tempsMax:
                temps = raw_input(msg)
                try:
                    temps = int(temps)
                except:
                    pass
        else:
            temps = tempsMax
    else:
        msg = "\nVous avez demande %i heures 'temps humain' de calcul." \
            % (TEMPS_MAX_H, )
        print msg

        msg = "\nDe combien de temps souhaitez-vous disposer"
        msg += " (le temps doit être inferieur ou egal a"
        msg += " %i heures)" % (tempsMax,)
        msg += " ?\n>>> "
        temps = tempsMax + 1
        while not temps <= tempsMax:
            temps = raw_input(msg)
            try:
                temps = int(temps)
            except:
                pass
else:
    if not isinstance(TEMPS_MAX_H, int):
        temps = tempsMax
    else:
        temps = min(tempsMax, TEMPS_MAX_H)

# Récapitulatif
print "\nVous avez demande un calcul :"
print "- de type : %s ;" % (jobtype,)
print "- sur %i noeuds ;" % (noeuds,)
print "- avec %i sockets ;" % (sockets,)
print "- avec %i coeurs physiques ;" % (coeurs,)
print "- avec %i Gb de memoire vive globale ;" % (ramGlobale,)
print "  - soit %i Gb de memoire par noeud ;" % (ramParNoeud,)
print "  - soit %i Gb de memoire par socket ;" % (ramParSocket,)
print "  - soit %i Gb de memoire par coeur ;" % (ramParCoeur,)
print "- pendant %i heures en 'temps humain'." % (temps,)







# =============================================================================
# TROISIÈME PARTIE : FICHIER CMD
# =============================================================================

# Génération du fichier scriptEnv.py et sauvegarde dans dossier spécifique ----
# La copie du fichier dans un dossier spécifique au job est essentielle
# car slurm étant appelé plus tard, si plusieurs ordres de calcul
# sont en attente, il faut que le fichier soit disponible au moment
# de l'exécution -- ultérieure -- du calcul (il ne faut pas que le fichier
# ait été détruit par un autre ordre de calcul entre temps).
genScriptEnv(MACHINES_FILE, ENV_FILE, coeursParNoeud)
os.system("mkdir " + inputFileName[:-4] + "_ScriptEnv")
os.system("cp scriptEnv.py ./" + inputFileName[:-4] + "_ScriptEnv")
os.system("rm scriptEnv.py")


# Commande ABAQUS -------------------------------------------------------------
if ABAQUS_EXEC != "abq2022hf1":
    command = "./"
else:
    command = ""

command += ABAQUS_EXEC \
        + " interactive" \
        + " job=" + inputFileName[:-4] \
        + " input=" + inputFileName

if fortranFile:
    command += " user=" + fortranFileName[:-2]

if restart:
    command += " oldjob=" + inputFileName[:-4]

command += " cpus=" + str(coeurs)
command += " output_precision=full"

if ABAQUS_JOB == "explicit":
    command += " double=both" \
             + " parallel=domain" \
             + " domains=" + str(coeurs) \
             + " mp_mode=MPI"

elif ABAQUS_JOB == "standard":
    command += " double=both" \
             + " mp_mode=MPI" \
             + " standard_parallel=all"
else:
    raise ValueError, 'ABAQUS_JOB est soit "explicit", soit "standard".'

command += " scratch=$WD"

if LOG_PERFORMANCE.lower() == "o":
    command += " >> " + LOG_PERFORMANCE_FILE

if jobtype == "mono":
    coeursParNoeud = coeurs
    ramParNoeud = ramParCoeur * coeurs

if jobtype == "mesca":
    print("\nIndiquez combien de mémoire vive vous souhaitez utiliser (en Go).")
    print("Vous pouvez monter jusqu'à 256 Go par socket")
    print("sachant qu'il y a 8 sockets de 16 coeurs chacun.")
    ramParNoeud = 0
    while not ramParNoeud > 0:
        ramParNoeud = raw_input(">>> ")
        try:
            ramParNoeud = int(ramParNoeud)
        except:
            ramParNoeud = 0
    ramParNoeud *= 1000

# Préparation du contenu à écrire dans le fichier SLURM -----------------------
msgSlurm = "#!/bin/bash\n"
msgSlurm += "#SBATCH -J " + inputFileName[:-4][:10] + "\n"
msgSlurm += "#SBATCH -N " + str(noeuds) + "\n"
msgSlurm += "#SBATCH -n " + str(coeurs) + "\n"
msgSlurm += "#SBATCH --time=%02i:00:00\n" % (temps, )

if jobtype in ("mono", "mesca"):
    msgSlurm += "#SBATCH --ntasks-per-node=" + str(coeurs) + "\n"
else:
    msgSlurm += "#SBATCH --ntasks-per-node=" + str(coeursParNoeud) + "\n"

msgSlurm += "#SBATCH --ntasks-per-core=1\n"
msgSlurm += "#SBATCH --mail-user=" + E_MAIL + "\n"

if jobtype == "mesca":
    msgSlurm += "#SBATCH --qos=mesca\n"
    msgSlurm += "#SBATCH --mem=" + str(ramParNoeud) + "\n"

msgSlurm += "\n"

msgTmp = ABAQUS_MOD[ABAQUS_EXEC][0]
for cmd in ABAQUS_MOD[ABAQUS_EXEC][1:]:
    msgTmp = msgTmp + "\n" + cmd

msgSlurm += msgTmp + "\n" * 2
msgSlurm += "export OMP_NUM_THREADS=1\n"
msgSlurm += "export HERE=`pwd`\n"
#msgSlurm += "\n"
#msgSlurm += "module show abaqus/2016"
msgSlurm += "cd $HERE\n"
msgSlurm += "\n"
msgSlurm += "node_list=`scontrol show hostname $SLURM_NODELIST | sort | uniq`\n"
msgSlurm += 'echo "$node_list" >> ' + MACHINES_FILE
msgSlurm += "\n" * 2

if len(USER_RELATIVE_FOLDER_FOR_WORKDIR) > 0:
    msgSlurm += "export WD=/tmpdir/$LOGNAME/"
    msgSlurm += USER_RELATIVE_FOLDER_FOR_WORKDIR
    msgSlurm += "/$SLURM_JOBID\n"
else:
    #msgSlurm += "export WD=/tmpdir/$LOGNAME/$SLURM_JOBID\n" # pour spécifier un dossier relatif en dur
    msgSlurm += "export WD=$HERE/$SLURM_JOBID\n" # pour utiliser le dossier courant

msgSlurm += "mkdir $WD\n"
msgSlurm += "\n"
msgSlurm += "cp $HERE/" + inputFileName + " $WD\n"
msgSlurm += "cp $HERE/" + inputFileName[:-3] + "cmd $WD\n"
msgSlurm += "cp $HERE/" + MACHINES_FILE + " $WD\n"
msgSlurm += "cp $HERE/" + inputFileName[:-4] + "_ScriptEnv/scriptEnv.py $WD\n"

msgSlurm += "sleep 1\n"
msgSlurm += "rm $HERE/" + MACHINES_FILE + "\n"
msgSlurm += "rm $HERE/" + inputFileName[:-4] + "_ScriptEnv/scriptEnv.py\n"
msgSlurm += "rm -R $HERE/" + inputFileName[:-4] + "_ScriptEnv\n"

if fortranFile:
    msgSlurm += "cp $HERE/" + fortranFileName + " $WD\n"

for otherFileName in otherFilesToCopy:
    msgSlurm += "cp $HERE/" + otherFileName + " $WD\n"

msgSlurm += "\n"
msgSlurm += "cd $WD\n"

#if noeuds > 1:
msgSlurm += "python scriptEnv.py\n"
msgSlurm += "sleep 1\n"

msgSlurm += "ln -s " + ABAQUS_PATH + "/" + ABAQUS_EXEC
msgSlurm += " " + ABAQUS_EXEC + "\n" * 2

msgSlurm += command
msgSlurm += "\n"


# Écriture du fichier pour SLURM ----------------------------------------------
with open(inputFileName[:-3] + "cmd", "w") as myFile:
    myFile.write(msgSlurm)


# Option d'affichage et de lancement automatique du script --------------------
if AFFICHER:
    choix = AFFICHER
else:
    choix = None
    msg = "\nVoulez-vous afficher le fichier " + inputFileName[:-4]
    msg += ".cmd (o/n) ?\n>>> "
while not choix in ("o", "n"):
    choix = raw_input(msg)
    choix = choix.lower()

if choix == "o":
    print "\n" + "*" * LARGEUR_TERMINAL
    os.system("cat " + inputFileName[:-3] + "cmd\n")
    print "*" * LARGEUR_TERMINAL

if EXECUTER:
    choix = EXECUTER
else:
    choix = None
    msg = "\nSouhaitez-vous executer sbatch " + inputFileName[:-3]
    msg += "cmd (o/n) ?\n>>> "
while not choix in ("o", "n"):
    choix = raw_input(msg)
    choix = choix.lower()

msg = "\n" + "=" * LARGEUR_TERMINAL
if choix == "n":
    msg += "\nLe job - N'A PAS - ete soumis.\n"
elif choix == "o":
    msg += "\nLe job - A - ete soumis.\n"
msg += "." * LARGEUR_TERMINAL
msg += "\nRappels :\n"
if choix == "n":
    msg += "- pour lancer le job: sbatch " + inputFileName[:-3] + "cmd\n"
msg += "- pour connaitre l'etat d'avancement du job : squeue -u LOGIN\n"
msg += "- pour annuler un job                       : scancel JOB_ID\n"
msg += "=" * LARGEUR_TERMINAL + "\n"
print msg

if choix == "o":
    os.system("sbatch " + inputFileName[:-3] + "cmd")
